package ch.frankel.blog.openid;

import static org.openid4java.message.ax.AxMessage.OPENID_NS_AX;

import java.io.IOException;
import java.util.List;

import javax.servlet.ServletConfig;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.openid4java.consumer.ConsumerException;
import org.openid4java.consumer.ConsumerManager;
import org.openid4java.consumer.VerificationResult;
import org.openid4java.discovery.DiscoveryInformation;
import org.openid4java.discovery.Identifier;
import org.openid4java.message.AuthRequest;
import org.openid4java.message.AuthSuccess;
import org.openid4java.message.MessageExtension;
import org.openid4java.message.ParameterList;
import org.openid4java.message.ax.FetchRequest;
import org.openid4java.message.ax.FetchResponse;

/**
 * Google authentication servlet.
 * 
 * @author <a href="http://blog.frankel.ch/">Nicolas Frankel</a>
 */
public class OpenIdServlet extends HttpServlet {

	/** Serial version unique identifier. */
	private static final long serialVersionUID = 1L;

	private ConsumerManager manager;

	@Override
	public void init(ServletConfig config) throws ServletException {

		super.init(config);

		try {

			manager = new ConsumerManager();

		} catch (ConsumerException e) {

			throw new ServletException(e);
		}
	}

	@Override
	protected void doGet(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {

		ParameterList openidResp = new ParameterList(request.getParameterMap());

		DiscoveryInformation discovered = (DiscoveryInformation) request.getSession().getAttribute("discovered");

		StringBuffer receivingURL = request.getRequestURL();

		String queryString = request.getQueryString();

		if (queryString != null && queryString.length() > 0) {

			receivingURL.append("?").append(request.getQueryString());
		}

		try {

			VerificationResult verification = manager.verify(receivingURL.toString(), openidResp, discovered);

			Identifier verified = verification.getVerifiedId();

			if (verified != null) {

				AuthSuccess authSuccess = (AuthSuccess) verification.getAuthResponse();

				if (authSuccess.hasExtension(OPENID_NS_AX)) {

					MessageExtension ext = authSuccess.getExtension(OPENID_NS_AX);

					if (ext instanceof FetchResponse) {

						FetchResponse fetch = (FetchResponse) ext;

						request.setAttribute("EMAIL", fetch.getAttributeValue("Email"));
						request.setAttribute("FIRST_NAME", fetch.getAttributeValue("FirstName"));
						request.setAttribute("LAST_NAME", fetch.getAttributeValue("LastName"));
						request.setAttribute("COUNTRY", fetch.getAttributeValue("Country"));
						request.setAttribute("LANG", fetch.getAttributeValue("Lang"));
					}
				}

				request.getRequestDispatcher("/WEB-INF/page/welcome.jsp").forward(request, response);

			} else {

				request.setAttribute("ERROR", "Authentication failed: " + verification.getStatusMsg());

				request.getRequestDispatcher("/").forward(request, response);
			}

		} catch (Exception e) {

			throw new ServletException(e);
		}
	}

	protected void doPost(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {

		try {

			String url = request.getParameter("url");
			
			List<?> discoveries = manager.discover(url);

			DiscoveryInformation discovered = manager.associate(discoveries);

			request.getSession().setAttribute("discovered", discovered);

			AuthRequest authReq = manager.authenticate(discovered, request.getRequestURL().toString());

			FetchRequest fetch = FetchRequest.createFetchRequest();

			fetch.addAttribute("Email", "http://schema.openid.net/contact/email", true);
			fetch.addAttribute("FirstName", "http://axschema.org/namePerson/first", true);
			fetch.addAttribute("LastName", "http://axschema.org/namePerson/last", true);
			fetch.addAttribute("Country", "http://axschema.org/contact/country/home", true);
			fetch.addAttribute("Lang", "http://axschema.org/pref/language", true);

			authReq.addExtension(fetch);

			response.sendRedirect(authReq.getDestinationUrl(true));

		} catch (Exception e) {

			throw new ServletException(e);
		}
	}
}
