package ch.frankel.blog.hibernate;

import static org.testng.Assert.assertEquals;

import java.io.File;
import java.net.URL;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.SQLSyntaxErrorException;

import org.apache.commons.io.FileUtils;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.cfg.AnnotationConfiguration;
import org.testng.annotations.BeforeMethod;
import org.testng.annotations.BeforeTest;
import org.testng.annotations.Test;

import ch.frankel.blog.hibernate.model.Person;

/**
 * Test class to check some basic truths about Hibernate.
 * 
 * @author Nicolas Frnkel
 * @since 14 juin 2009
 */
public class Tests {

    /** Hibernate's session factory. */
    private SessionFactory factory;

    /**
     * Start with a fresh DB.
     * 
     * @throws java.lang.Exception
     */
    @BeforeTest
    public void setUpBeforeClass() throws Exception {

        // Register within the driver manager
        Class.forName("org.apache.derby.jdbc.EmbeddedDriver");

        Connection connection = DriverManager.getConnection("jdbc:derby:target/myDb;create=true");

        try {

            try {

                PreparedStatement dropStmt = connection.prepareStatement("DROP TABLE PERSON");

                dropStmt.execute();

            } catch (SQLSyntaxErrorException e) {

                // Silent catch since we don't care
            }

            URL url = getClass().getClassLoader().getResource("ddl.sql");

            File file = new File(url.toURI());

            String sql = FileUtils.readFileToString(file);

            PreparedStatement ddlStmt = connection.prepareStatement(sql);

            ddlStmt.execute();

        } finally {

            connection.close();
        }
    }

    /**
     * Sets up the factory used for the test.
     * 
     * @throws Exception
     * 
     */
    @SuppressWarnings("unused")
    @BeforeMethod
    private void setUp() throws Exception {

        factory = new AnnotationConfiguration().configure().buildSessionFactory();

        Connection connection = DriverManager.getConnection("jdbc:derby:target/myDb;create=true");

        try {

            URL url = getClass().getClassLoader().getResource("dml.sql");

            File file = new File(url.toURI());

            String sql = FileUtils.readFileToString(file);

            PreparedStatement dmlStmt = connection.prepareStatement(sql);

            dmlStmt.execute();

        } finally {

            connection.close();
        }
    }

    /**
     * Test class for checking whether you need to explicitly call the
     * <code>update()</code> method on a persistent object.
     * 
     * @throws SQLException
     */
    @Test
    public void testUpdateWithoutCall() throws SQLException {

        Session session = factory.getCurrentSession();

        session.getTransaction().begin();

        Person person = (Person) session.load(Person.class, 1L);

        // Change the value without calling update
        person.setLastName("FooBar");

        session.getTransaction().commit();

        long id = person.getId();

        Connection connection = DriverManager.getConnection("jdbc:derby:target/myDb");

        PreparedStatement stmt = connection.prepareStatement("SELECT LAST_NAME FROM PERSON WHERE ID = ?");

        stmt.setLong(1, id);

        ResultSet rs = stmt.executeQuery();

        rs.next();

        String lastName = rs.getString("LAST_NAME");

        connection.close();

        assertEquals(lastName, "FooBar");
    }
}
